/************************************************************************/
/*									*/
/*	LIBSGL.C				(C) 1/94 Ch. Muenkel	*/
/*	Main part of SGL library -- basic subroutines			*/
/*									*/
/************************************************************************/
/*									*/
/*	Simple Graphics Library						*/
/*	An easy to use interface to the X11 window system		*/
/*									*/
/************************************************************************/
/*									*/
/*	Author:	Christian Muenkel					*/
/*		Present address:					*/
/*		muenkel@carbo.tphys.uni-heidelberg.de			*/
/*		Christian Muenkel					*/
/*		Institut fuer Theoretische Physik			*/
/*		Philosophenweg 19					*/
/*		69120 Heidelberg - Germany				*/
/*									*/
/************************************************************************/
/* version      date            comment                                 */
/*                                                                      */
/* V1.0		 1/94		first implementation 			*/
/* V1.1		 4/94		z-buffer / Gouraud-shading		*/
/*				shading: draw in Image (not pixmap)	*/
/*                                                                      */
/************************************************************************/
/*                                                                      */
/* To do:								*/
/*                                                                      */
/************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<math.h>

#define		__SGL_MAIN
#include	"sglglob.h"
#undef		__SGL_MAIN




/************************************************************************/
/************************************************************************/
/*									*/
/* INITIALIZATIONS ...							*/
/*									*/
/************************************************************************/
/************************************************************************/

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_init(void)						*/
/*		initialize X11 graphics					*/
/*									*/
/************************************************************************/
void	sgl_init(void)
{
  if (flag_initialized == True)
     return;
  sgl_init_nowin();
  sgl_create_win(256,256);
}


/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_init_nowin(void)					*/
/*		initialize X11 graphics					*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_init_nowin(void)
{
  int		i;
  XVisualInfo	visinf;
  char		str[1024];

  if (flag_initialized == True)
     return;

  X11GD == NULL;
  X11GD3D == NULL;
  GD_act = GD_num = 0;


  X11CD->color_pixel=(unsigned long *)calloc(COLOR_MAX,sizeof(unsigned long));
  X11CD->color_red=(unsigned short *)calloc(COLOR_MAX,sizeof(unsigned short));
  X11CD->color_green=(unsigned short *)calloc(COLOR_MAX,
	sizeof(unsigned short));
  X11CD->color_blue=(unsigned short *)calloc(COLOR_MAX,sizeof(unsigned short));
  if (X11CD->color_pixel==NULL || X11CD->color_red==NULL ||
	X11CD->color_green==NULL || X11CD->color_blue==NULL)
     FatalError("SGL_INIT: Can't allocate color data structure.\n");
  X11CD->color_background = 0;
  X11CD->color_foreground = 1;
  X11CD->color_stack_top = 2;
  sgl_create_colmap();

  flag_initialized = True;

  sgl3d_init();

}


/************************************************************************/
/************************************************************************/
/*									*/
/* DRAWING FUNCTIONS							*/
/*									*/
/************************************************************************/
/************************************************************************/

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_line(x1,y1,x2,y2)					*/
/*		draw a line from (x1,y1) to (x2,y2) in viewport coord.	*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_line(float x1, float y1, float x2, float y2)
{
  int	ix1,iy1,ix2,iy2;

  ix1 = ((x1 - X11GD->win_x_min) * X11GD->win_idx) * X11GD->win_sx;
  ix2 = ((x2 - X11GD->win_x_min) * X11GD->win_idx) * X11GD->win_sx;
  iy1 = X11GD->win_sy - ((y1 - X11GD->win_y_min) * X11GD->win_idy) * X11GD->win_sy;
  iy2 = X11GD->win_sy - ((y2 - X11GD->win_y_min) * X11GD->win_idy) * X11GD->win_sy;

  if (flag_initialized == True)
    XDrawLine(X11GD->disp,X11GD->pm,X11GD->gc,ix1,iy1,ix2,iy2);
}

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_circle(x,y,r)					*/
/*		draw a circle at (x,y) with radius r in viewport coord	*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_circle(float x, float y, float r)
{
  int	ix,iy,ir;

  ix = (((x-r) - X11GD->win_x_min) * X11GD->win_idx) * X11GD->win_sx;
  iy = X11GD->win_sy - (((y+r) - X11GD->win_y_min) * X11GD->win_idy) * X11GD->win_sy;
  ir = (r * X11GD->win_idx) * X11GD->win_sx * 2;

  if (flag_initialized == True)
    XDrawArc(X11GD->disp,X11GD->pm,X11GD->gc,ix,iy,ir,ir,0,64*360);
}

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_disc(x,y,r)						*/
/*		draw a filled circle at (x,y) with radius r in viewport	*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_disc(float x, float y, float r)
{
  int	ix,iy,ir;

  ix = (((x-r) - X11GD->win_x_min) * X11GD->win_idx) * X11GD->win_sx;
  iy = X11GD->win_sy - (((y+r) - X11GD->win_y_min) * X11GD->win_idy) * X11GD->win_sy;
  ir = (r * X11GD->win_idx) * X11GD->win_sx * 2;

  if (flag_initialized == True)
    XFillArc(X11GD->disp,X11GD->pm,X11GD->gc,ix,iy,ir,ir,0,64*360);
}

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_rectangle(x,y,dx,dy)				*/
/*		draw a rectangle; (x,y) is the lower-right corner	*/
/*		(dx,dy) are the width/ height of the rectangle		*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_rectangle(float x, float y, float dx, float dy)
{
  int	ix,iy,idx,idy;

  ix  = ((x - X11GD->win_x_min) * X11GD->win_idx) * X11GD->win_sx;
  idx = (dx * X11GD->win_idx) * X11GD->win_sx;
  iy  = X11GD->win_sy - (((y+dy) - X11GD->win_y_min) * X11GD->win_idy) * X11GD->win_sy;
  idy = (dy * X11GD->win_idy) * X11GD->win_sy;

  if (flag_initialized == True)
    XDrawRectangle(X11GD->disp,X11GD->pm,X11GD->gc,ix,iy,idx,idy);
}

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_rectarea(x,y,dx,dy)					*/
/*		draw a filled rectangle; (x,y) is the lower-right	*/
/*		corner, (dx,dy) are the width/ height of the rectangle	*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_rectarea(float x, float y, float dx, float dy)
{
  int	ix,iy,idx,idy;

  ix  = ((x - X11GD->win_x_min) * X11GD->win_idx) * X11GD->win_sx;
  idx = (dx * X11GD->win_idx) * X11GD->win_sx;
  iy  = X11GD->win_sy - (((y+dy) - X11GD->win_y_min) * X11GD->win_idy) * X11GD->win_sy;
  idy = (dy * X11GD->win_idy) * X11GD->win_sy;

  if (flag_initialized == True)
    XFillRectangle(X11GD->disp,X11GD->pm,X11GD->gc,ix,iy,idx,idy);
}

  

/************************************************************************/
/************************************************************************/
/*									*/
/* Modifiers 								*/
/*									*/
/************************************************************************/
/************************************************************************/

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_winsize(sx,sy)					*/
/*		redefine window size in pixels				*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_winsize(int new_sx, int new_sy)
{
  if (flag_initialized != True)
     return;
 sgl_delete_win();
 sgl_priv_create_win(new_sx,new_sy,-1);
}


/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_viewport(xmin,ymin,dx,dy)				*/
/*		redefine viewport coordinate system			*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_viewport(float xmin, float ymin, float dx, float dy)
{ 
  int	flag,new_sx,new_sy;

  if (fabs((X11GD->win_idy/X11GD->win_idx) - (dx/dy)) > 1E-3)
    flag = 1;				/* window aspect ratio changed	*/ 
  else
    flag = 0;
  X11GD->win_x_min = xmin;
  X11GD->win_y_min = ymin;
  X11GD->win_x_max = xmin + dx;
  X11GD->win_y_max = ymin + dy;
  X11GD->win_idx = 1.0 / dx;
  X11GD->win_idy = 1.0 / dy;
  new_sx = X11GD->win_sx;
  new_sy = X11GD->win_sx * (dy/dx);
  if (flag != 0)
    sgl_winsize(new_sx,new_sy);
}


/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_display(void)					*/
/*		display background drawings (double buffer)		*/
/*									*/
/************************************************************************/
/************************************************************************/
void		sgl_display(void)
{
  int	i;
  unsigned char	*ptr;

  if (X11GD3D->shading == SGL3D_LINES)
   {	XCopyArea(X11GD->disp,X11GD->pm,X11GD->win,
		X11GD->gc,0,0,X11GD->win_sx,X11GD->win_sy,0,0);
	XSetForeground( X11GD->disp, X11GD->gc,
		X11CD->color_pixel[X11CD->color_background]);
	XFillRectangle(X11GD->disp, X11GD->pm,
		X11GD->gc, 0,0,X11GD->win_sx,X11GD->win_sy);
	XSetForeground( X11GD->disp, X11GD->gc,
		X11CD->color_pixel[X11CD->color_foreground]);
	XSync(X11GD->disp, X11GD->win);
   }
  else if (X11GD3D->shading == SGL3D_GOURAUD)
   {	XPutImage(X11GD->disp,X11GD->win,X11GD->gc,
		X11GD->xim,0,0,0,0,
		X11GD->win_sx,X11GD->win_sy);
	XSync(X11GD->disp, X11GD->win);
        ptr = (unsigned char *) X11GD->imdata;
        for (i = X11GD->win_sy*X11GD->xim->bytes_per_line-1; i >= 0; i--)
          ptr[i] = 0;
	sgl3d_zclear();
   }
  else
   { fprintf(stderr,"SGL_DISPLAY: Unknown shading type.\n");
   }

}


/************************************************************************/
/************************************************************************/
/*									*/
/* COLOR FUNCTIONS							*/
/*									*/
/************************************************************************/
/************************************************************************/

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_create_colmap(void)					*/
/*		create (pseudo) color colormap				*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_create_colmap(void)
{
  XColor	exactColor,actualColor;
  int		ret;
  char		*disp_name,str[256];
  XVisualInfo   visinf;

  disp_name = NULL;
  X11CD->disp = XOpenDisplay( disp_name );
  if ( ! X11CD->disp ) 
   { sprintf(str,"Can't open display %s .\n",XDisplayName(disp_name));
     FatalError(str);
   }
  X11CD->scrnn = DefaultScreen( X11CD->disp );
  X11CD->scrn  = ScreenOfDisplay(X11CD->disp, X11CD->scrnn);

  /***** Detect color model used by display ******************************/
  /***** first, check for best: TrueColor display ************************/
  if ( XMatchVisualInfo( X11CD->disp, X11CD->scrnn,24, TrueColor, &visinf))
    X11CD->vistype = VISTYPE_TrueColor; 
  /***** second, check for PseudoColor display ************************/
  else if (XMatchVisualInfo( X11CD->disp, X11CD->scrnn,8,PseudoColor,&visinf))
    X11CD->vistype = VISTYPE_PseudoColor;
  /***** third, check for StaticGray, GrayScale display ****************/
  else if (XMatchVisualInfo( X11CD->disp, X11CD->scrnn,1,StaticGray,&visinf))
    X11CD->vistype = VISTYPE_StaticGray;
  else if (XMatchVisualInfo( X11CD->disp, X11CD->scrnn,1,GrayScale,&visinf))
    X11CD->vistype = VISTYPE_GrayScale;
  else
    FatalError("SGL_INIT: Sorry, this type of Visual can't be used.\n");

  X11CD->vis = visinf.visual;
  X11CD->depth = visinf.depth;

  /***** Use default colormap if possible *****/
  if (X11CD->vis == DefaultVisualOfScreen(X11CD->scrn) )
   { X11CD->colmap = DefaultColormapOfScreen( X11CD->scrn );
   }
  else
   { X11CD->colmap = XCreateColormap(X11CD->disp,
		RootWindow(X11CD->disp,X11CD->scrnn),X11CD->vis,AllocNone);
   }
  if (X11CD->colmap == 0) 
    FatalError("SGL_CREATE_COLMAP: Can't create color map for visual.\n");

/*
  ret = XAllocNamedColor(X11GD->disp, X11GD->colmap, "black",
	&actualColor,&exactColor);
  if (ret == 0)
    fprintf(stderr,"sgl_create_colmap: Can't allocate 'black' color.\n");
  X11GD->color_pixel[0] = actualColor.pixel;
  ret = XAllocNamedColor(X11GD->disp, X11GD->colmap, "white",
	&actualColor,&exactColor);
  if (ret == 0)
    fprintf(stderr,"sgl_create_colmap: Can't allocate 'white' color.\n");
  X11GD->color_pixel[1] = actualColor.pixel;
  X11GD->color_stack_top = 2;
*/
  X11CD->color_pixel[0]= BlackPixel(X11CD->disp,X11CD->scrnn);
  X11CD->color_pixel[1]= WhitePixel(X11CD->disp,X11CD->scrnn);
  X11CD->color_stack_top = 2;
}

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_color(int)						*/
/*		choose actual drawing color				*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_color(int colindex)
{
  if (flag_initialized != True)
   { fprintf(stderr,"WARNING: sgl not initialized.\n");
     return;
   }
  switch(X11GD->vistype)
   { case VISTYPE_GrayScale:
	break;
     case VISTYPE_StaticGray:
	break;
     case VISTYPE_PseudoColor:
     case VISTYPE_TrueColor:
	if (colindex < 0)
	  fprintf(stderr,"SGL_COLOR: There are no negativ (%d) colors.\n",colindex);
	else if (colindex >= X11CD->color_stack_top)
	  fprintf(stderr,"SGL_COLOR: color %d (of %d)not defined .\n",
		colindex,X11CD->color_stack_top);
	else
	 { XSetForeground(X11GD->disp,X11GD->gc,X11CD->color_pixel[colindex]);
	   X11CD->color_foreground = colindex;
           X11CD->rgb_color[0] = (float)X11CD->color_red[colindex] / 65535.0;
           X11CD->rgb_color[1] = (float)X11CD->color_green[colindex] / 65535.0;
           X11CD->rgb_color[2] = (float)X11CD->color_blue[colindex] / 65535.0;
	 }
	break;
   }
 XSync(X11GD->disp, X11GD->win);
}

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_color_get_rgb(float,float,float)			*/
/*		get (r,g,b) triple of actual color			*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_color_get_rgb(float *red, float *green, float *blue)
{
  *red   = X11CD->rgb_color[0];
  *green = X11CD->rgb_color[1];
  *blue  = X11CD->rgb_color[2];
}

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_color_background(int)				*/
/*		choose actual background color				*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_color_background(int colindex)
{
  if (flag_initialized != True)
   { fprintf(stderr,"WARNING: sgl not initialized.\n");
     return;
   }
  switch(X11GD->vistype)
   { case VISTYPE_GrayScale:
	break;
     case VISTYPE_StaticGray:
	break;
     case VISTYPE_PseudoColor:
     case VISTYPE_TrueColor:
	if (colindex < 0)
	  fprintf(stderr,"SGL_COLOR_BACKGROUND: There are no negativ (%d) colors.\n",colindex);
	else if (colindex >= X11CD->color_stack_top)
	  fprintf(stderr,"SGL_COLOR_BACKGROUND: color %d not defined.\n",colindex);
	else
	 { XSetBackground(X11GD->disp,X11GD->gc,X11CD->color_pixel[colindex]);
	   X11CD->color_background = colindex;
	 }
	break;
   }
  XSetForeground(X11GD->disp,X11GD->gc,
	X11CD->color_pixel[X11CD->color_background]);
  XFillRectangle(X11GD->disp,X11GD->pm,X11GD->gc,0,0,
		X11GD->win_sx,X11GD->win_sy);
  XSetForeground(X11GD->disp,X11GD->gc,
		X11CD->color_pixel[X11CD->color_foreground]);
  XSync(X11GD->disp, X11GD->win);
}

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_colordef(int, char *)				*/
/*		define a colormap entry					*/
/*									*/
/************************************************************************/
/************************************************************************/
int	sgl_colordef(char *colname)
{
  XColor	exactColor,actualColor;
  int		ret,return_value;

  if (flag_initialized != True)
   { fprintf(stderr,"WARNING: sgl was not initialized.\n");
     return(-1);
   }

  return_value = -1;
  switch(X11GD->vistype)
   { case VISTYPE_GrayScale:
	return_value = 1;
	break;
     case VISTYPE_StaticGray:
	return_value = 1;
	break;
     case VISTYPE_TrueColor:
     case VISTYPE_PseudoColor:
/*
	XLookupColor( X11GD->disp,
		DefaultColormapOfScreen(X11GD->scrn),colname,
		&exactColor,&actualColor);
	X11GD->color_pixel[color_stack_top] = actualColor.pixel;
	XSetForeground( X11GD->disp, X11GD->gc, 
			X11GD->color_pixel[color_stack_top] );
fprintf(stderr,"color_def set %x\n",actualColor.pixel);
	color_foreground = color_stack_top;
	color_stack_top++;
	return_value = (color_stack_top-1);
	break;
*/
	ret = XAllocNamedColor( X11GD->disp, X11CD->colmap,
		 colname,&actualColor,&exactColor );
	if (ret == 0)
	 { fprintf(stderr,"SGL_COLORDEF: Can't allocate color (%s).\n",
		colname);
	   return(-1);
	 }
	else
	 { X11CD->color_pixel[X11CD->color_stack_top] = actualColor.pixel;
           X11CD->color_red[X11CD->color_stack_top] = actualColor.red;
           X11CD->color_green[X11CD->color_stack_top] = actualColor.green;
           X11CD->color_blue[X11CD->color_stack_top] = actualColor.blue;
	   XSetForeground( X11GD->disp, X11GD->gc, 
			X11CD->color_pixel[X11CD->color_stack_top] );
	   if (X11CD->color_stack_top < COLOR_MAX)
	    { X11CD->color_stack_top++;
	      return_value = (X11CD->color_stack_top-1);
	    }
	 }
	break;
    }
  return(return_value);
}
 


/************************************************************************/
/************************************************************************/
/*									*/
/* MISC. ...								*/
/*									*/
/************************************************************************/
/************************************************************************/


/************************************************************************/
/************************************************************************/
/*									*/
/*  void	sgl_debugging(int)					*/
/*		set flag, print debugging information or not		*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl_debugging(int value)
{
 if (value == 0)	debugging = False;
 else			debugging = True;
}

/************************************************************************/
/************************************************************************/
/*									*/
/*  void	FatalError(char *)					*/
/*		printf fatal error message and exit			*/
/*									*/
/************************************************************************/
/************************************************************************/
void	FatalError(char *str)
{
 fprintf(stderr,"%s",str);
 exit(-1);
}

